/* -*-c++-*-
 * This source code is proprietary of ADIT
 * Copyright (C) 2013 Advanced Driver Information Technology Joint Venture GmbH
 * All rights reserved
 *
 * Author: Vadiraj Kaamsha <vadiraj.kaamsha@in.bosch.com>
 * Author: Rudolf Dederer <rudolf.dederer@de.bosch.com>
*/

#include <limits.h>
#include <osg/Depth>
#include <algorithm>
#include <osg/FrameBufferObject>
#include <osgUtil/Statistics>
#include <osgBatchedText/BatchElementBase>
#include <osgBatchedText/BatcherBase>
#include <osgBatchedText/TextSimpleBase>

using namespace osgBatchedText;

void BatchedVertexAttributes::init(bool useHalfFloat)
{
   if (useHalfFloat)
   {
      _vertices = new VecArray<osg::Vec4h>();
   }
   else
   {
      _vertices = new VecArray<osg::Vec4f>();
   }
}

void BatchedVertexAttributes::clear()
{
   _vertices->clear();
   _texCoord.clear();
   _colorArray.clear();
   _transVecAttribOutline.clear();
   _relOffset.clear();
}

void BatchedVertexAttributes::reserve(unsigned int num)
{
   _vertices->reserve(num);
   _texCoord.reserve(num);
   _colorArray.reserve(num);
   _transVecAttribOutline.reserve(num);
   _relOffset.reserve(num);
}

void BatchedVertexAttributes::resize(unsigned int num)
{
   _vertices->resize(num);
   _texCoord.resize(num);
   _colorArray.resize(num);
   _transVecAttribOutline.resize(num);
   if (_relOffset.size() > 0)
   {
      _relOffset.resize(num);
   }
}

BatchDrawableBase::BatchDrawableBase(BatcherBase* parent, BatchDrawableBase::tShaderType shaderType, bool enableDepthTest)
   : _parent(parent), _shaderType(shaderType), _enableDepthTest(enableDepthTest)
{
   init(_parent->getUseHalfFloat());
}

BatchDrawableBase::BatchDrawableBase(const BatchDrawableBase& src, const osg::CopyOp& copyop)
   : osg::Drawable(src, copyop), _parent(src._parent), _shaderType(src._shaderType), _enableDepthTest(src._enableDepthTest)
{
   bool useHalfFloat = (src._vertAttrib._vertices->getDataTypeSize() == sizeof(osg::Vec4h));
   init(useHalfFloat);
}

void BatchDrawableBase::init(bool useHalfFloat)
{
   setStateSet(_parent->getFontStateSet());
   setUseDisplayList(false);
   setSupportsDisplayList(false);
   //Create Float/Half Float Arrays
   _vertAttrib.init(useHalfFloat);
}

void BatchDrawableBase::sortListForDrawing() const
{
   if (_parent->getSortBasedOnDisplayOrder())
   {
      //sort based on display priority
      std::stable_sort(_lstBatchElementContainer.begin(), _lstBatchElementContainer.end(), BatchElementContainer::sortBasedOnDisplayOrder);
   }
   else
   {
      //sort based on z-distance. Using stable_sort to get a deterministic comparison as there is no control on the input floating point values.
      std::stable_sort(_lstBatchElementContainer.begin(), _lstBatchElementContainer.end(), BatchElementContainer::z_comparator);
   }
}

void BatchDrawableBase::drawImplementation(osg::RenderInfo& renderInfo) const
{
   //check view
   if (_parent->getCurrentView() != renderInfo.getView())
   {
      clearElementsToDraw();
      return;
   }

   osg::State& state = *renderInfo.getState();
   _parent->initializeTexture(state);

   _parent->releaseDeletedTextures(state);

   if (0 < _lstBatchElementContainer.size())
   {
      calculateTransVec(state.getModelViewMatrix());

      reserveVertexAttribMemory();

      _parent->updateUniforms(state);

      std::vector< osg::ref_ptr<BatchElementContainer> >::const_iterator elemItr = _lstBatchElementContainer.begin();
      while (elemItr != _lstBatchElementContainer.end())
      {
         prepareVertextAttributes(state, elemItr);

         draw(state);
         _vertAttrib.clear();
         _parent->resetTextureModified();
      }
   }
}

unsigned int BatchDrawableBase::getNumberOfVertices() const
{
   unsigned int numVertices = 0;
   for (std::vector< osg::ref_ptr<BatchElementContainer> >::const_iterator itr = _lstBatchElementContainer.begin(); itr != _lstBatchElementContainer.end(); ++itr)
   {
      numVertices += (*itr)->getNumberOfVertices();
   }
   return numVertices;
}


void BatchDrawableBase::reserveVertexAttribMemory() const
{
   if (0 < _lstBatchElementContainer.size())
   {
      //sort based on z
      sortListForDrawing();

      _vertAttrib.reserve(getNumberOfVertices());
   }
}

void BatchDrawableBase::bindAttribLocations(osg::Program* shaderProgram)
{
   // Set up some per vertex attributes here.
   shaderProgram->addBindAttribLocation("a_TransVecOutline", _transVecOutlineAttribLoc);
   shaderProgram->addBindAttribLocation("a_RelOffset", _relOffset);
}

void BatchDrawableBase::draw(osg::State &state) const
{
   if (!_vertAttrib._vertices->empty())
   {
      glDepthMask(GL_FALSE);
      state.haveAppliedAttribute(osg::StateAttribute::DEPTH);
      state.disableAllVertexArrays();
      state.applyMode(GL_BLEND, true);
      state.setColorPointer(4, GL_FLOAT, (sizeof(osg::Vec4)), &(_vertAttrib._colorArray.front()));

      if (_vertAttrib._vertices->getDataTypeSize() == sizeof(osg::Vec4h))
      {
         state.setVertexPointer(4, GL_HALF_FLOAT, 0, &(_vertAttrib.getVertexArrayHalfFloat()->asVector().front()));
      }
      else
      {
         state.setVertexPointer(4, GL_FLOAT, 0, &(_vertAttrib.getVertexArrayFloat()->asVector().front()));
      }

      if (!_vertAttrib._texCoord.empty() && (_vertAttrib._texCoord.size() >= _vertAttrib._vertices->size())) state.setTexCoordPointer(0, 3, GL_FLOAT, 0, &(_vertAttrib._texCoord.front()));
      state.setVertexAttribPointer(_transVecOutlineAttribLoc, 4, GL_FLOAT, GL_FALSE, 0, &(_vertAttrib._transVecAttribOutline.front()));

      if (!_vertAttrib._relOffset.empty())
      {
         state.setVertexAttribPointer(_relOffset, 3, GL_FLOAT, GL_FALSE, 0, &(_vertAttrib._relOffset.front()));
      }

      if (NULL != _parent->getTextureObject(E_INNER_FONT_TEXTURE))
      {
         state.setActiveTextureUnit(E_INNER_FONT_TEXTURE);
         _parent->getTextureObject(E_INNER_FONT_TEXTURE)->bind();
      }

      if (_parent->getOutline() && (NULL != _parent->getTextureObject(E_OUTER_FONT_TEXTURE)))
      {
         state.setActiveTextureUnit(E_OUTER_FONT_TEXTURE);
         _parent->getTextureObject(E_OUTER_FONT_TEXTURE)->bind();
      }

      state.drawQuads(0, _vertAttrib._vertices->size());
   }
}

bool BatchDrawableBase::prepareVertextAttributes(osg::State& state, std::vector< osg::ref_ptr<BatchElementContainer> >::const_iterator& elemItr) const
{
   BatchDrawParams batchDrawParams(this, &state);

   _parent->incCounter();

   std::vector< osg::ref_ptr<BatchElementContainer> >::const_iterator elemItrOld = elemItr;
   unsigned int nextElementIndexOld = (*elemItr)->_renderInfo._nextElementIndex;

   while (elemItr != _lstBatchElementContainer.end())
   {
      if ((*elemItr)->setVertexAttributes(_vertAttrib, batchDrawParams)
       || ((elemItrOld == elemItr) && ((*elemItr)->_renderInfo._nextElementIndex == nextElementIndexOld)))
      {
         ++elemItr;
      }
      else
      {
         break;
      }
   }

   if (true == _parent->isMipmapActive())
   {
      _parent->generateMipMap(state);
   }

   return true;
}

void BatchDrawableBase::accept(osg::Drawable::ConstAttributeFunctor& af) const
{
   if (_vertAttrib._vertices->size() > 0)
   {
      if (_vertAttrib._vertices->getDataTypeSize() == sizeof(osg::Vec4h))
      {
         af.apply(osg::Drawable::VERTICES,_vertAttrib._vertices->size(), &(_vertAttrib.getVertexArrayHalfFloat()->asVector().front()));
      }
      else
      {
         af.apply(osg::Drawable::VERTICES,_vertAttrib._vertices->size(), &(_vertAttrib.getVertexArrayFloat()->asVector().front()));
      }
      af.apply(osg::Drawable::TEXTURE_COORDS_0,_vertAttrib._texCoord.size(), &(_vertAttrib._texCoord.front()));
   }
}

void BatchDrawableBase::accept(osg::PrimitiveFunctor& pf) const
{
   osgUtil::Statistics* stats = dynamic_cast<osgUtil::Statistics*>(&pf);
   if (NULL != stats)
   {
      unsigned int numVertices = getNumberOfVertices();
      stats->drawElements(GL_TRIANGLES, numVertices, (GLubyte*)NULL);
      stats->setVertexArray(numVertices, (osg::Vec3*)NULL);
   }
   else
   {
      osg::Drawable::accept(pf);
   }
}

void BatchDrawableBase::setThreadSafeRefUnref(bool threadSafe)
{
    Drawable::setThreadSafeRefUnref(threadSafe);

    _parent->getActiveFont()->setThreadSafeRefUnref(threadSafe);
}

void BatchDrawableBase::resizeGLObjectBuffers(unsigned int maxSize)
{
    Drawable::resizeGLObjectBuffers(maxSize);

    _parent->getActiveFont()->resizeGLObjectBuffers(maxSize);
}

void BatchDrawableBase::releaseGLObjects(osg::State* state) const
{
    Drawable::releaseGLObjects(state);
    _parent->getActiveFont()->releaseGLObjects(state);
}

void BatchDrawableBase::calculateTransVec(const osg::Matrix& mvMat) const
{
   for (std::vector< osg::ref_ptr<BatchElementContainer> >::iterator itr = _lstBatchElementContainer.begin(); itr != _lstBatchElementContainer.end(); ++itr)
   {
      (*itr)->calcTransVector(mvMat);
   }
}

void BatchDrawableBase::addBatchElementContainer(const osg::ref_ptr<BatchElementContainer>& batchElementContainer, bool update)
{
   std::pair< std::set<BatchElementContainer*, batchElementContainerPtrCmp>::iterator, bool> ret = _setBatchElementContainer.insert(batchElementContainer.get());

   if (ret.second || !update)
   {
      _lstBatchElementContainer.push_back(batchElementContainer);
   }
   else
   {
      //If already exists, just add the alpha value to the existing element
      float& alpha = (*(ret.first))->_renderInfo._alpha;
      alpha = osg::minimum((batchElementContainer->_renderInfo._alpha + alpha), 1.f);
   }
}


bool BatchDrawableBase::batchElementContainerPtrCmp::operator() (const BatchElementContainer* const& lhs, const BatchElementContainer* const& rhs) const
{
   return *lhs < *rhs;
}
